import _ from 'lodash'

function isDef(val) {
  return val !== undefined && val !== null
}

function getFirstComponentChild(children) {
  if (Array.isArray(children)) {
    for (let i = 0; i < children.length; i++) {
      const c = children[i]
      if (isDef(c) && isDef(c.componentOptions)) {
        return c
      }
    }
  }
}

function remove(arr, item) {
  if (arr.length) {
    const index = arr.indexOf(item)
    if (index > -1) {
      return arr.splice(index, 1)
    }
  }
}

function getComponentName(opts) {
  return opts && (opts.Ctor.options.name || opts.tag)
}

function matches(pattern, name) {
  if (Array.isArray(pattern)) {
    return pattern.indexOf(name) > -1
  } else if (typeof pattern === 'string') {
    return pattern.split(',').indexOf(name) > -1
  } else if (_.isRegExp(pattern)) {
    return pattern.test(name)
  }
  /* istanbul ignore next */
  return false
}

function pruneCache(cache, current, filter) {
  for (const key in cache) {
    const cachedNode = cache[key]
    if (cachedNode) {
      const name = getComponentName(cachedNode.componentOptions)
      if (name && !filter(name)) {
        if (cachedNode !== current) {
          pruneCacheEntry(cachedNode, [], key)
        }
        cache[key] = null
      }
    }
  }
}

function pruneCacheEntry(vnode, keys, key) {
  if (vnode) {
    vnode.componentInstance.$destroy()
  }
  remove(keys, key)
}

export default {
  name: 'sweet-keep-alive',

  props: {
    include: [],
    exclude: [],
    updateComponentsKey: Function,
    max: [String, Number]
  },

  created() {
    // vue的keep-alive存储对象
    this.cache = Object.create(null)
    this.keys = []
  },
  // 调用keep-alive组件销毁钩子，组件销毁的时候同时清除缓存
  destroyed() {
    for (const key in this.cache) {
      pruneCacheEntry(this.cache[key], this.keys, key)
    }
  },

  watch: {
    include(val) {
      pruneCache(this.cache, this._vnode, name => matches(val, name))
    },
    exclude(val) {
      pruneCache(this.cache, this._vnode, name => !matches(val, name))
    }
  },

  render() {
    const vnode = getFirstComponentChild(this.$slots.default)
    const componentOptions = vnode && vnode.componentOptions
    if (componentOptions) {
      // check pattern
      const name = getComponentName(componentOptions)
      if (name && (
        (this.include && !matches(this.include, name)) ||
        (this.exclude && matches(this.exclude, name))
      )) {
        return vnode
      }
      const key = vnode.key == null
        // same constructor may get registered as different local components
        // so cid alone is not enough (#3269)
        ? componentOptions.Ctor.cid + (componentOptions.tag ? `::${componentOptions.tag}` : '')
        : vnode.key
      // 添加获取key的外部hook
      this.updateComponentsKey && this.updateComponentsKey(key)
      if (this.cache[key]) {
        vnode.componentInstance = this.cache[key].componentInstance

        // remove(this.keys, key)
        // this.keys.push(key)
      } else {
        this.cache[key] = vnode
        this.keys.push(key)
        if (this.max && this.keys.length > parseInt(this.max)) {
          pruneCacheEntry(this.cache[this.keys[0]], this.keys, key)
        }
      }
      vnode.data.keepAlive = true
    }
    return vnode
  },
  methods: {
    // 通过cache的key删除对应的缓存
    removeCacheByKey(key) {
      pruneCacheEntry(this.cache[key], this.keys, key)
      this.cache[key] = null
    }
  }
}
